<?php

namespace App\Http\Controllers;

use App\Models\BlockIp;
use App\Models\Domain;
use App\Models\EmailTemplate;
use App\Models\Order;
use App\Models\Script;
use App\Models\Setting;
use App\Models\SmsApi;
use App\Models\SmsTemplate;
use App\Models\Smtp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class EmailOrderController extends Controller
{
    private $prefix = "admin/email-order/";
    private $resources = [];
    private $checkPermissionAndRolesController;
    private $smtpController;
    private $emailTemplateController;

    public function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController,
        SmtpController $smtpController,
        EmailTemplateController $emailTemplateController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
        $this->smtpController = $smtpController;
        $this->emailTemplateController = $emailTemplateController;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Manage Email Order']);
        $this->resources['data'] = $this->getEmailOrders($request, 15);
        $settings = Setting::where('user_id', $request->user()->id)->first();
        $this->resources['settings'] = ($settings) ? $settings : [];
        return view('admin.email-orders.email_orders_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = Order::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('type', 'email')
                    ->where(function ($query) use ($request) {
                        $query->where('imei', 'like', '%' . $request->search . '%')
                            ->orWhere('email', 'like', '%' . $request->search . '%')
                            ->orWhere('phone_number', 'like', '%' . $request->search . '%')
                            ->orWhere('model', 'like', '%' . $request->search . '%')
                            ->orWhere('link', 'like', '%' . $request->search . '%');
                    });

                if (!$superAdmin) {
                    $order->where('created_by', $request->user()->id);
                }
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            $settings = Setting::where('user_id', $request->user()->id)->first();
            $this->resources['settings'] = ($settings) ? $settings : [];
            return view('admin.email-orders.email_orders_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Email Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['smtp'] = $this->getSmtpList($request);
        $this->resources['email_templates'] = $this->getEmailTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        return view('admin.email-orders.email_orders_add')->with($this->resources);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Email Order']);
        $credentials = $request->validate([
            'domain_id' => ['required'],
            'imei' => ['required'],
            'email' => ['required'],
            'model' => ['required'],
            'script_id' => ['required'],
        ]);

        if ($request->without_email ==  "true") {
            $send_email = 1;
        } else {
            $credentials = $request->validate([
                'smtp_id' => ['required'],
                'template_id' => ['required'],
                'subject' => ['required'],
                'from_name' => ['required']
            ]);
            $send_email = 0;
        }
        if ($request->custom_link_code != null || $request->custom_link_code != "") {
            $request->validate([
                'custom_link_code' => ['min:5'],
            ]);
        }

        try {
            $storeParam = $request->except('_token', 'without_sms', 'custom_link');
            $storeParam['created_by'] = $request->user()->id;
            $storeParam['without_email'] = $send_email;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['link_code_length']) && $settings['order_settings']['link_code_length'] != false) {
                $length = $settings['order_settings']['link_code_length'];
            } else {
                $length = 3;
            }
            $storeParam['code'] = $this->unique_code($length);
            $storeParam['type'] = 'email';
            if ($request->custom_link != '') {
                $storeParam['link'] = $request->custom_link;
            }
            DB::beginTransaction();
            $order = Order::create($storeParam);
            if ($request->custom_link == '') {
                $link = $this->generateLink($request, $order, $request->custom_link_code);
                $storeParam['link'] = $link;
            }
            DB::commit();
            if ($request->without_email !=  "true") {
                $this->sendMailPhpMailer($order->id);
            }
            $this->resources['messages'] = array("type" => "success", "description" => "Email Order creation success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function view(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'View Email Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['smtp'] = $this->getSmtpList($request);
        $this->resources['email_templates'] = $this->getEmailTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        $this->resources['data'] = $this->getEmailOrderById($request);
        return view('admin.email-orders.email_orders_view')->with($this->resources);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function edit(Order $order, Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Update Email Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['smtp'] = $this->getSmtpList($request);
        $this->resources['email_templates'] = $this->getEmailTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        $this->resources['data'] = $this->getEmailOrderById($request);
        return view('admin.email-orders.email_orders_edit')->with($this->resources);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Order $order)
    {

        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Email Order']);
        $credentials = $request->validate([
            'id' => ['required'],
            'domain_id' => ['required'],
            'imei' => ['required'],
            'email' => ['required'],
            'model' => ['required'],
            'script_id' => ['required'],
        ]);
        if ($request->without_email ==  "true") {
            $send_email = 1;
        } else {
            $credentials = $request->validate([
                'smtp_id' => ['required'],
                'template_id' => ['required'],
                'subject' => ['required'],
                'from_name' => ['required']
            ]);
            $send_email = 0;
        }
        if ($request->custom_link_code != null || $request->custom_link_code != "") {
            $request->validate([
                'custom_link_code' => ['min:5'],
            ]);
        }

        try {
            $storeParam = $request->except('_token', 'without_sms', 'id', 'custom_link');
            $storeParam['created_by'] = $request->user()->id;
            $storeParam['without_email'] = $send_email;
            if ($request->custom_link != '') {

                $storeParam['link'] = $request->custom_link;
            }
            DB::beginTransaction();
            $order_id = Order::where('id', $request->id)->update($storeParam);
            DB::commit();
            if ($request->custom_link == '') {
                $this->updateLink($request, $request->id, $request->custom_link_code);
            }
            $this->resources['messages'] = array("type" => "success", "description" => "Email Order update success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Delete Email Order']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $request->id)->where('type', 'email');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Order deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Order deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function getEmailOrders($request, $count)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::orderBy('id', 'desc')->where('type', 'email');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function getEmailOrderById($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $request->id)->where('type', 'email');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function getAuthorizedDomainList($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
            if (!$superAdmin) {
                $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
            }
            $result = $query->select('domains.*')->get();
            return $result;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getSmtpList($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Smtp::orderBy('id', 'desc')->select('name', 'id');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->get();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function getEmailTemplates($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::orderBy('id', 'desc')->select('name', 'id');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->get();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function getScripts()
    {
        try {
            $result = Script::all();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function generateLink($request, $order, $custom_link_code)
    {

        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
        if (!$superAdmin) {
            $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
        }
        $result = $query->where('domains.id', $order->domain_id)->select('domains.link')->first()->toArray();
        if ($custom_link_code != null) {
            $order_code = $custom_link_code;
            Order::where('id', $order->id)->update([
                'code' => $order_code,
            ]);
        } else {
            $order_code = $order->code;
        }
        if (!empty($result)) {
            $domain = $result['link'];
            // $script = Script::where('id', $order->script_id)->first();
            // $path = $script->path;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['default_link_type']) && $settings['order_settings']['default_link_type'] != false) {
                $link = $domain . '/' . $settings['order_settings']['default_link_type'] . $order_code;
            } else {
                $link = $domain . '/' . $order_code;
            }
            Order::where('id', $order->id)->update([
                "link" => $link
            ]);
            return $link;
        } else {
            return false;
        }
    }

    public function updateLink($request, $order_id, $custom_link_code)
    {
        $order = Order::where('id', $order_id)->first();
        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
        if (!$superAdmin) {
            $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
        }
        $result = $query->where('domains.id', $order->domain_id)->select('domains.link')->first()->toArray();
        if ($custom_link_code != null) {
            $order_code = $custom_link_code;
            Order::where('id', $order->id)->update([
                'code' => $order_code,
            ]);
        } else {
            $order_code = $order->code;
        }
        if (!empty($result)) {
            $domain = $result['link'];
            // $script = Script::where('id', $order->script_id)->first();
            // $path = $script->path;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['default_link_type']) && $settings['order_settings']['default_link_type'] != false) {
                $link = $domain . '/' . $settings['order_settings']['default_link_type'] . $order_code;
            } else {
                $link = $domain . '/' . $order_code;
            }
            Order::where('id', $order->id)->update([
                "link" => $link
            ]);
            return $link;
        } else {
            return false;
        }
    }

    public function block(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $query->update([
                "blocked" => 1
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Order link blocked success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function unBlock(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $query->update([
                "blocked" => 0
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Order link unblocked success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function getEmailTemplateAjax(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return response()->json(['template' => $result->template], 200);
        } catch (\Throwable $th) {
            return response()->json($th->getMessage(), 500);
        }
    }
    function unique_code($limit)
    {
        // Ensure limit is between 3 and 20
        $limit = max(3, min(20, $limit));

        // Characters pools
        $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $lowercase = 'abcdefghijklmnopqrstuvwxyz';
        $numbers = '0123456789';

        // Shuffle and select characters
        $upper = $uppercase[mt_rand(0, strlen($uppercase) - 1)];
        $lower = $lowercase[mt_rand(0, strlen($lowercase) - 1)];
        $number = $numbers[mt_rand(0, strlen($numbers) - 1)];

        // Start with one of each type for the minimum condition
        $code = $upper . $lower . $number;

        // Combine all character sets for the remaining code
        $allCharacters = $uppercase . $lowercase . $numbers;
        $remainingLength = $limit - 3;
        for ($i = 0; $i < $remainingLength; $i++) {
            $code .= $allCharacters[mt_rand(0, strlen($allCharacters) - 1)];
        }

        // Shuffle to randomize character positions
        $code = str_shuffle($code);

        return $code;
    }

    public function SelectDelete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Delete Email Order']);

        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!empty($request->delete)) {
                foreach ($request->delete as $value) {

                    $query = Order::where('id', $value);
                    if (!$superAdmin) {
                        $query->where('created_by', $request->user()->id);
                    }
                    $result = $query->delete();
                }
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Order deleted failed");
            }

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Selected Order deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Order deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function MailConfig(array $data)
    {
        try {
            if (!empty($data)) {
                Config::set('mail.driver', $data['driver']);
                Config::set('mail.host', $data['host_name']);
                Config::set('mail.port', $data['port']);
                Config::set('mail.username', $data['username']);
                Config::set('mail.password', $data['password']);
                Config::set('mail.encryption', $data['encryption']);
                return true;
            } else {
                return false;
            }
        } catch (\Throwable $th) {
            dd($th);
        }
    }

    public function sendMail($id)
    {
        try {
            $data = Order::where('id', $id)->first()->toArray();
            $smtp = Smtp::where('id', $data['smtp_id'])->first()->toArray();
            $set_config = $this->MailConfig([
                'driver' => $smtp['driver'],
                'host_name' => $smtp['host_name'],
                'port' => $smtp['port'],
                'username' => $smtp['username'],
                'password' => $smtp['password'],
                'encryption' => $smtp['encrypted_type']
            ]);
            if ($set_config) {
                $template = EmailTemplate::where('id', $data['template_id'])->first()->toArray();
                $mail_body = str_replace('[VIEW_LOCATION', $data['link'], $template['template']);
                $data = array_merge((array) $data, array("email_body" => $mail_body));
                $data = array_merge((array) $data, array("smtp" => $smtp));
                Mail::send('admin.emails.email_order', ['data' => $data], function ($message) use ($data) {
                    $message->from($data['smtp']['username'], $data['from_name']);
                    $message->to($data['email']);
                    $message->subject($data['subject']);
                });
                // Log::channel('script')->info(' - sendSuccessMail() $e', [$e]);
            } else {
                dd("config set failed");
            }
        } catch (\Exception $e) {
            dd($e);
        }
    }
    public function sendMailPhpMailer($id)
    {

        require base_path("vendor/autoload.php");
        $mail = new PHPMailer(true);     // Passing `true` enables exceptions
        $data = Order::where('id', $id)->first()->toArray();
        $smtp = Smtp::where('id', $data['smtp_id'])->first()->toArray();
        $template = EmailTemplate::where('id', $data['template_id'])->first()->toArray();
        $mail_body = str_replace('[VIEW_LOCATION]', $data['link'], $template['template']);
        try {

            // Email server settings
            $mail->SMTPDebug = 0;
            $mail->isSMTP();
            $mail->Host = $smtp['host_name'];             //  smtp host
            $mail->SMTPAuth = true;
            $mail->Username = $smtp['username'];   //  sender username
            $mail->Password = $smtp['password'];       // sender password
            $mail->SMTPSecure = $smtp['encrypted_type'];                  // encryption - ssl/tls
            $mail->Port = $smtp['port'];                          // port - 587/465

            $mail->setFrom($smtp['username'], $data['from_name']);
            $mail->addAddress($data['email']);
            // $mail->addCC($request->emailCc);
            // $mail->addBCC($request->emailBcc);

            // $mail->addReplyTo('sender-reply-email', 'sender-reply-name');

            // if(isset($_FILES['emailAttachments'])) {
            //     for ($i=0; $i < count($_FILES['emailAttachments']['tmp_name']); $i++) {
            //         $mail->addAttachment($_FILES['emailAttachments']['tmp_name'][$i], $_FILES['emailAttachments']['name'][$i]);
            //     }
            // }
            $mail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );

            $mail->isHTML(true);                // Set email content format to HTML

            $mail->Subject = $data['subject'];
            $mail->Body    = $mail_body;

            // $mail->AltBody = plain text version of email body;

            if (!$mail->send()) {
                // return back()->with("failed", "Email not sent.")->withErrors($mail->ErrorInfo);
            } else {
                // dd('success');
                // return back()->with("success", "Email has been sent.");
            }
        } catch (\Exception $e) {
            // dd($e);
            //  return back()->with('error','Message could not be sent.');
        }
    }
}
