<?php

namespace App\Http\Controllers;

use App\Models\Smtp;
use Illuminate\Http\Request;

class SmtpController extends Controller
{
    private $prefix = "admin/smtp/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }

    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage SMTP']);
        $this->resources['data'] = $this->getSmtp($request, 15);
        return view('admin.smtp.smtp_manage')->with($this->resources);
    }
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create SMTP']);
        return view('admin.smtp.smtp_add')->with($this->resources);
    }
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Create SMTP']);

        $credentials = $request->validate([
            'name' => ['required'],
            'host_name' => ['required'],
            'port'=>['required'],
            'driver'=>['required'],
            'username'=>['required'],
            'password'=>['required'],
            'encrypted_type'=>['required'],
        ]);
        try {

            $storeParam = $request->except('_token');

            $storeParam['created_by'] = $request->user()->id;
            Smtp::create($storeParam);
            $this->resources['messages'] = array("type" => "success", "description"=>"Smtp creation success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function view(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'View SMTP']);
        $this->resources['data'] = $this->getSmtpById($request);
        return view('admin.smtp.smtp_view')->with($this->resources);
    }
    public function edit(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Edit SMTP']);
        $this->resources['data'] = $this->getSmtpById($request);
        return view('admin.smtp.smtp_edit')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = Smtp::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
                if(!$superAdmin){
                    $order->where('created_by', $request->user()->id);
                }
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.smtp.smtp_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Update SMTP']);

        $credentials = $request->validate([
            'id'=>['required'],
            'name' => ['required'],
            'host_name' => ['required'],
            'port'=>['required'],
            'driver'=>['required'],
            'username'=>['required'],
            'password'=>['required'],
            'encrypted_type'=>['required'],
        ]);
        try {

            $storeParam = $request->except('_token');
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Smtp::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->update($storeParam);
            if($result){
                $this->resources['messages'] = array("type" => "success", "description"=>"Smtp updated success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"Smtp updated failed");
            }
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Delete SMTP']);
        $request->validate([
            'id' => ['required'],   
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Smtp::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if($result){
                $this->resources['messages'] = array("type" => "success", "description"=>"Smtp deleted success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"Smtp deleted failed");
            }
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function getSmtp($request, $count)
    {
        try{
           
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Smtp::orderBy('id','desc');
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function getSmtpById($request)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Smtp::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
}
