<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SubscriptionController extends Controller
{
    private $prefix = "admin/subscription/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $this->resources['users'] = User::select('name', 'id')->get();
        return view('admin.subscription.add')->with($this->resources);
    }
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $credentials = $request->validate([
            'user' => ['required'],
            'from' => ['required'],
            'to' => ['required'],
        ]);
        try {
            $requestParam = $request->except('_token');
            $subscription = $this->getSubscriptionByUser($requestParam['user']);
            if (isset($subscription['expired']) && $subscription['expired'] != 1) {
                $this->resources['messages'] = array("type" => "error", "description" => "User already subscribed");
                return redirect()->back()->with('messages', $this->resources);
            }
            $currentTime = Carbon::now()->format('H:i:s');
            Subscription::create([
                'user' => $requestParam['user'],
                'from' => $requestParam['from'] . ' ' . $currentTime,
                'to' => $requestParam['to'] . ' ' . $currentTime,
                'expired' => 0
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "User Subscription creation success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function getSubscriptionByUser($user_id)
    {
        try {
            return Subscription::where('user', $user_id)->first();
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function getSubscriptionById($id)
    {
        try {
            return Subscription::where('id', $id)->first();
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function manage(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $this->resources['data'] = Subscription::join('users', 'subscriptions.user', 'users.id')->select('users.name as user_name', 'subscriptions.*')->paginate(15);
        return view('admin.subscription.manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = Subscription::join('users', 'subscriptions.user', 'users.id');
            if ($request->search) {
                $order->where('users.name', 'Like', '%' . $request->search . '%');
            }
            $this->resources['data'] = $order->select('users.name as user_name', 'subscriptions.*')->orderBy('subscriptions.id', 'DESC')->paginate(15);
            return view('admin.subscription.manage')->with($this->resources);
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function edit(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $this->resources['data'] = $this->getSubscriptionById($request->id);
        $this->resources['users'] = User::select('name', 'id')->get();
        return view('admin.subscription.edit')->with($this->resources);
    }

    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $credentials = $request->validate([
            'id' => ['required'],
            'from' => ['required'],
            'to' => ['required'],
        ]);
        try {
            $requestParam = $request->all();
            $currentTime = Carbon::now()->format('H:i:s');

            Subscription::where('id', $requestParam['id'])->update([
                'from' => $requestParam['from'] . ' ' . $currentTime,
                'to' => $requestParam['to'] . ' ' . $currentTime,
                'notify' => 0,
                'expired' => 0
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "User Subscription updated success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Subscription']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $result = Subscription::where('id', $request->id)->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Subscription deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Subscription deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
}
