<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\WBReplyText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class WBReplyTextController extends Controller
{
    private $prefix = "admin/whatsapp-bot/reply-text/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Bot']);
        $this->resources['data'] = $this->get($request);
        return view('admin.whatsapp-robot.wsr_replytext')->with($this->resources);
    }
    public function get($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!$superAdmin) {
                $result = WBReplyText::where('created_by', Auth::user()->id)->paginate(15);
            } else {
                $result = WBReplyText::paginate(15);
            }
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Bot']);
        $credentials = $request->validate([
            'name' => ['required'],
            'message' => ['required'],
            'keyword' => ['required'],
            'text' => ['required']
        ]);
        try {
            $requestAll = $request->all();
            $options = [];
            foreach ($requestAll['keyword'] as $key => $value) {
                $options[] = ["keyword" => $value, "text" => $requestAll['text'][$key]];
            }
            WBReplyText::create([
                "name" => $requestAll['name'],
                "message" => $requestAll['message'],
                "options" => $options,
                "created_by" => Auth::user()->id
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Reply text creation success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function edit(Request $request)
    {
        $credentials = $request->validate([
            'id' => ['required']
        ]);
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Bot']);
        $this->resources['data'] = $this->getById($request);
        return view('admin.whatsapp-robot.wsr_replytext_edit')->with($this->resources);
    }

    public function getById($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!$superAdmin) {
                $result = WBReplyText::where('id', $request->id)->where('created_by', Auth::user()->id)->first();
            } else {
                $result = WBReplyText::where('id', $request->id)->first();
            }
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Bot']);
        $credentials = $request->validate([
            'id' => ['required'],
            'name' => ['required'],
            'message' => ['required'],
            'keyword' => ['required'],
            'text' => ['required']
        ]);
        try {
            $requestAll = $request->all();
            $options = [];
            foreach ($requestAll['keyword'] as $key => $value) {
                $options[] = ["keyword" => $value, "text" => $requestAll['text'][$key]];
            }
            WBReplyText::where('id', $requestAll['id'])->update([
                "name" => $requestAll['name'],
                "message" => $requestAll['message'],
                "options" => $options
            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Reply text updated success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Bot']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WBReplyText::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Reply text deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Reply text deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function getReplyText(Request $request) {
        try {
            $requestAll = $request->all();
            
            $order_id = $requestAll['order_id'];
            $result = Order::join('w_b_reply_texts', 'orders.reply_text_id', 'w_b_reply_texts.id')->where('orders.id', $order_id)->select('w_b_reply_texts.message','w_b_reply_texts.options','orders.link')->first();
            $data["message"] = $result['message'];
            $options = json_decode($result['options'], true);
            foreach($options as $item){
                $data["options"][] = ["keyword"=>$item['keyword'], "text"=> str_replace(["[LINK]"], $result["link"], $item["text"])];
            }
            return response()->json($data, 200);
        } catch (\Throwable $th) {
            Log::info("getReplyText", [$th->getMessage() . ' ||| ' . $th->getLine()]);
            return response()->json($th->getMessage() . ' ||| ' . $th->getLine(), 500);
        }
    }
}
