<?php

namespace App\Http\Controllers;

use App\Models\WhatsAppSenderDevice;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WhatsAppSenderDeviceController extends Controller
{
    private $prefix = "admin/whatsapp-sender/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $this->resources['data'] = $this->get($request);
        return view('admin.whatsapp-sender.ws_add_view_device')->with($this->resources);
    }
    public function orderindex(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Manage WhatsApp Order']);
        $this->resources['data'] = $this->get($request);
        return view('admin.whatsapp-sender.order.orders_manage')->with($this->resources);
    }
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $credentials = $request->validate([
            'name' => ['required'],
            'number' => ['required'],
            'multi_device' => ['required'],
            'message_delay' => ['required']
        ]);
        try {
            $this->resources['data'] = $this->get($request);
            $result = $this->create_session();
            if ($result) {
                $requestParam = $request->all();
                $requestParam['created_by'] = Auth::user()->id;
                $requestParam['status'] = 'initiate';
                $requestParam['data'] = $result;
                $requestParam['qr_update_time'] = now();
                WhatsAppSenderDevice::create($requestParam);
                $this->resources['messages'] = array("type" => "success", "description" => "Device creation success");
                return redirect()->back()->with('messages', $this->resources);
            } else {
                $this->resources['messages'] = array("type" => "success", "description" => "Device session creation success");
                return redirect()->back()->with('messages', $this->resources);
            }
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function get($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!$superAdmin) {
                $result = WhatsAppSenderDevice::where('created_by', Auth::user()->id)->paginate(15);
            } else {
                $result = WhatsAppSenderDevice::paginate(15);
            }
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function create_session()
    {
        $data = [
            "id" => $this->generateJobId(15),
            "isLegacy" => false,
            "domain" => "isender.shop"
        ];

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => 'https://whatsapp.igensolutions.limited/sessions/add',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/json'
            ),
        ));

        $response = curl_exec($curl);
        $info = curl_getinfo($curl);
        curl_close($curl);
        dd($response, $info);
        if ($info['http_code'] == 200) {
            return array("response" => $response, "request_data" => $data);
        } else {
            return false;
        }
    }
    public function check_session($session_id)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://whatsapp.igensolutions.limited/sessions/find/$session_id",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
        ));

        $response = curl_exec($curl);
        $info = curl_getinfo($curl);
        curl_close($curl);
        if ($info['http_code'] == 200) {
            return true;
        }
        return false;
    }
    public function delete_session($session_id)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://whatsapp.igensolutions.limited/sessions/delete/$session_id",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
        ));

        $response = curl_exec($curl);
        $info = curl_getinfo($curl);
        curl_close($curl);
        if ($info['http_code'] == 200) {
            return true;
        }
        return false;
    }
    /**
     * @return unique id
     */
    public function generateJobId($length = 10)
    {
        try {
            $id =  substr(base_convert(sha1(uniqid(mt_rand())), 16, 36), 0, $length);
            return strtoupper($id);
        } catch (\Throwable $th) {
            throw $th;
        }
    }
    public function edit(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $this->resources['data'] = $this->getById($request);
        return view('admin.whatsapp-sender.ws_edit')->with($this->resources);
    }

    public function getById($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppSenderDevice::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $credentials = $request->validate([
            'id' => ['required'],
            'name' => ['required'],
            'number' => ['required'],
            'multi_device' => ['required'],
            'message_delay' => ['required']
        ]);
        try {

            $requestParam = $request->all();
            WhatsAppSenderDevice::where('id', $requestParam['id'])->update([
                'name' => $requestParam['name'],
                'number' => $requestParam['number'],
                'multi_device' => $requestParam['multi_device'],
                'message_delay' => $requestParam['message_delay'],
                'description' => $requestParam['description'],

            ]);
            $this->resources['messages'] = array("type" => "success", "description" => "Device updated success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            dd($th);
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $data = $this->getById($request);
            if ($data['status'] != 'connected') {
                $session_id = $data['data']['request_data']['id'];
                $result = $this->delete_session($session_id);
            }
            $query = WhatsAppSenderDevice::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "WhatsApp Sender Device deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "WhatsApp Sender Device deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function scan(Request $request)
    {
        try {
            $id = $request->id;
            $check = $this->getById($request);
            if ($check['status'] != 'connected') {
                $result = $this->create_session();
                if ($result) {
                    $update = WhatsAppSenderDevice::where('id', $id)->update([
                        'data' => $result,
                        'qr_update_time' => now()
                    ]);
                    if ($update) {
                        $get_data = $this->getData($id);
                        $responseData = json_decode($get_data['data']['response'], true);
                        return response()->json(['qr' => $responseData['data']['qr']], 200);
                    }
                }
            }

            return response()->json("something vent wrong", 500);
        } catch (\Throwable $th) {
            return response()->json($th->getMessage(), 500);
        }
    }

    public function getData($id)
    {
        try {
            return WhatsAppSenderDevice::select('data', 'status')->where('id', $id)->first();
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function statusUpdate(Request $request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppSenderDevice::select('data', 'id', 'status');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->get();
            if ($result) {
                foreach ($result as $value) {
                    $session_id = $value['data']['request_data']['id'];
                    $result = $this->check_session($session_id);
                    if ($result) {
                        WhatsAppSenderDevice::where('id', $value['id'])->update([
                            'status' => 'connected'
                        ]);
                    } else {
                        if ($value['status'] == 'connected') {
                            WhatsAppSenderDevice::where('id', $value['id'])->update([
                                'status' => 'disconnected'
                            ]);
                        }
                    }
                }
            }
            return response()->json("true", 200);
        } catch (\Throwable $th) {
            return response()->json($th->getMessage(), 500);
        }
    }
    public function statusUpdateById(Request $request)
    {
        try {
            $requestAll = $request->all();
            $id = $requestAll['id'];
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppSenderDevice::select('data', 'id', 'status', 'qr_update_time')->where('id', $id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $data = $query->first();
            if ($data) {
                $session_id = $data['data']['request_data']['id'];
                $result = $this->check_session($session_id);
                if ($result) {
                    WhatsAppSenderDevice::where('id', $data['id'])->update([
                        'status' => 'connected'
                    ]);
                } else {
                    if ($data['status'] == 'connected') {
                        WhatsAppSenderDevice::where('id', $data['id'])->update([
                            'status' => 'disconnected'
                        ]);
                    }
                }
            } else {
                return response()->json(false, 404);
            }

            $qr_update_time = $data['qr_update_time'];
            $start = new DateTime($qr_update_time);
            $end = new DateTime(now());
            $diff = $start->diff($end);
            if ($diff->s > 20) {
                $result = $this->create_session();
                if ($result) {
                    $update = WhatsAppSenderDevice::where('id', $id)->update([
                        'data' => $result,
                        'qr_update_time' => now()
                    ]);
                    if ($update) {
                        $get_data = $this->getData($id);
                        $responseData = json_decode($get_data['data']['response'], true);
                        return response()->json(['qr' => $responseData['data']['qr'], 'status' => $get_data['status']], 200);
                    }
                }
            } else {
                $get_data = $this->getData($id);
                return response()->json(['qr' => "", 'status' => $get_data['status']], 200);
            }
            return response()->json($diff->s, 200);
        } catch (\Throwable $th) {
            return response()->json($th->getLine(), 500);

            return response()->json($th->getMessage(), 500);
        }
    }
    public function disconnect(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'WhatsApp Sender Devices']);
        $credentials = $request->validate([
            'id' => ['required'],
        ]);

        try {
            $requestParam = $request->all();
            $data = $this->getById($request);
            if ($data['status'] == 'connected') {
                $session_id = $data['data']['request_data']['id'];
                $result = $this->delete_session($session_id);
                if ($result) {
                    WhatsAppSenderDevice::where('id', $requestParam['id'])->update([
                        'statu' => 'disconnected',
                    ]);
                    $this->resources['messages'] = array("type" => "success", "description" => "Device updated success");
                    return redirect()->back()->with('messages', $this->resources);
                }
            }
            $this->resources['messages'] = array("type" => "error", "description" => "Something went wrong!");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
}
