<?php

namespace App\Http\Controllers;

use App\Models\BlockIp;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BlockIpController extends Controller
{
    private $prefix = "admin/block-ip/";
    private $resources = [];
    private $checkPermissionAndRolesController;
    private $userController;

    public function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController,
        UserController $userController
    ) {

        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
        $this->userController = $userController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Block IP']);
        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $result = BlockIp::orderBy('id', 'desc')->get()->toArray();
        $data = [];
        foreach ($result as $item) {
            $data[] = $item['ips'];
        }
        $this->resources['data']  = $data;
        return view('admin.block-ip.index')->with($this->resources);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function manage(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Block IP']);
        $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
        $result = BlockIp::orderBy('id', 'desc')->where('user_id', Auth::user()->id)->paginate(15);
        $this->resources['data']  = $result;
        return view('admin.block-ip.block_ip_manage')->with($this->resources);
    }

    public function save(Request $request)
    {
        $credentials = $request->validate([
            'ips' => ['required'],
        ]);
        try {
            $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Block IP']);
            $data = explode(",", $request->ips);
            if (BlockIp::count() > 0) {
                BlockIp::where('user_id', $request->user()->id)->delete();
            }
            foreach ($data as $value) {
                BlockIp::create([
                    "ips" => trim($value),
                    "user_id" => $request->user()->id
                ]);
            }
            $this->resources['messages'] = array("type" => "success", "description" => "IP block list saved successfully");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function checkIpIsBlocked(Request $request)
    {
        try {
            $data = $request->all();
            $order = Order::where('code', $data['order_code'])->select('created_by')->first();
            if ($order) {
                $count = BlockIp::where('ips', $data['ip'])->where('user_id', $order->created_by)->count();
                if ($count > 0) {
                    return response(array('is_blocked' => true), 200);
                }
            }

            return response(array('is_blocked' => false), 200);
        } catch (\Exception $e) {
            throw $e;
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Block IP']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $result = BlockIp::where('id', $request->id)->where('user_id', Auth::user()->id)->delete();
            $this->resources['messages'] = array("type" => "success", "description" => "Ip deleted success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function SelectDelete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Block IP']);

        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if (!empty($request->delete)) {
                foreach ($request->delete as $value) {

                    $query = BlockIp::where('id', $value);
                    if (!$superAdmin) {
                        $query->where('user_id', $request->user()->id);
                    }
                    $result = $query->delete();
                }
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Ips deleted failed");
            }

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Selected Ips deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Selected Ips deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
}
