<?php

namespace App\Http\Controllers;

use App\Models\EmailTemplate;
use App\Models\Smtp;
use Illuminate\Http\Request;

class EmailTemplateController extends Controller
{
    private $prefix = "admin/email-templates/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    function __construct(
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
    ) {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }

    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Email Templates']);
        $this->resources['data'] = $this->getEmailTemplate($request, 15);
        return view('admin.email-templates.email_templates_manage')->with($this->resources);
    }
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create Email Templates']);
        return view('admin.email-templates.email_templates_add')->with($this->resources);
    }
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Create Email Templates']);

        $credentials = $request->validate([
            'name' => ['required'],
            'template' => ['required'],
        ]);
        try {

            $storeParam = $request->except('_token');

            $storeParam['created_by'] = $request->user()->id;
            EmailTemplate::create($storeParam);
            $this->resources['messages'] = array("type" => "success", "description"=>"Email Template creation success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function view(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'View Email Templates']);
        $this->resources['data'] = $this->getEmailTemplateById($request);
        return view('admin.email-templates.email_templates_view')->with($this->resources);
    }
    public function edit(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Edit Email Templates']);
        $this->resources['data'] = $this->getEmailTemplateById($request);
        return view('admin.email-templates.email_templates_edit')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = EmailTemplate::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
                if(!$superAdmin){
                    $order->where('created_by', $request->user()->id);
                }
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.email-templates.email_templates_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Update Email Templates']);

        $credentials = $request->validate([
            'id'=>['required'],
            'name' => ['required'],
            'template' => ['required'],
        ]);
        try {

            $storeParam = $request->except('_token');
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->update($storeParam);
            if($result){
                $this->resources['messages'] = array("type" => "success", "description"=>"Email Templates updated success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"Email Templates updated failed");
            }
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Delete Email Templates']);
        $request->validate([
            'id' => ['required'],   
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if($result){
                $this->resources['messages'] = array("type" => "success", "description"=>"Email Template deleted success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"Email Template deleted failed");
            }
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function getEmailTemplate($request, $count)
    {
        try{
           
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::orderBy('id','desc');
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function getEmailTemplateById($request)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = EmailTemplate::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
}
