<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;

class RoleController extends Controller
{
    private $prefix = "admin/users/roles/";
    private $resources = [];
    private $permissionController;
    private $checkPermissionAndRolesController;
    public function __construct(
        Request $request,
        PermissionController $permissionController,
        CheckPermissionsAndRolesController $checkPermissionAndRolesController
        )
    {
        $this->resources['prefix'] = $this->prefix;
        $this->permissionController = $permissionController;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);
        $this->resources['data'] = $this->getRoles(15);
        return view('admin.users.roles.roles_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = Role::query();
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.users.roles.roles_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function addRole(Request $request)
    {   
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);
        $this->resources['permissions'] = $this->permissionController->getAll();
        return view('admin.users.roles.role_add')->with($this->resources);
    }
    public function getRoles($count)
    {
        try{
            $result = Role::orderBy('id','desc')->paginate($count);
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function getRoleById($id)
    {
        try{
            $result = Role::where('id', $id)->first();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function store(Request $request)
    { 
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);

        $credentials = $request->validate([
            'guard_name' => ['required'],
            'name' => ['required'],
            'permissions'=>['required']
        ]);
        try {
            $storeParam = $request->except('_token','permissions');
            $role = Role::create($storeParam);
            if(isset($request->permissions) && !empty($request->permissions)){
                $role->syncPermissions($request->permissions);
            }
            $this->resources['messages'] = array("type" => "success", "description"=>"Role creation success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function editRole(Request $request)
    {   
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);
        $this->resources['role'] = $this->getRoleById($request->id);
        $this->resources['permissions'] = $this->permissionController->getAll();
        return view('admin.users.roles.role_edit')->with($this->resources);
    }
    public function update(Request $request)
    { 
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);
        $credentials = $request->validate([
            'guard_name' => ['required'],
            'name' => ['required'],
            'permissions'=>['required']
        ]);
        try {
            
            $storeParam = $request->except('_token','permissions');
            Role::where('id', $request->id)->update($storeParam);
            $role = Role::find($request->id);
            if(isset($request->permissions) && !empty($request->permissions)){
                $role->syncPermissions($request->permissions);
            }
            $this->resources['messages'] = array("type" => "success", "description"=>"Role updated success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }

    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Manage Roles']);
        $request->validate([
            'id' => ['required'],   
        ]);
        try {
            $result = Role::where('id', $request->id)->delete();
            $this->resources['messages'] = array("type" => "success", "description"=>"Role deleted success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
        
    }

    

}
