<?php

namespace App\Http\Controllers;

use App\Models\SmsApi;
use Illuminate\Http\Request;

class SmsApiController extends Controller
{
    private $prefix = "admin/sms-api/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    public function __construct(CheckPermissionsAndRolesController $checkPermissionAndRolesController)
    {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Manage SmsApi']);
        $this->resources['data'] = $this->getApi($request, 15);
        return view('admin.sms-api.sms_api_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {

            $request->validate([
                'search' => ['required'],
            ]);
            $order = SmsApi::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
                if (!$superAdmin) {
                    $order->where('created_by', $request->user()->id);
                }
                $order->orWhere('url', 'Like', '%' . $request->search . '%');
                $order->orWhere('method', 'Like', '%' . $request->search . '%');
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.sms-api.sms_api_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create SmsApi']);
        return view('admin.sms-api.sms_api_add')->with($this->resources);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create SmsApi']);

        $credentials = $request->validate([
            'name' => ['required'],
            'url' => ['required', 'url'],
            'method' => ['required'],
            'authentication' => ['required'],
            'json_encoded' => ['required'],
            'api_type' => ['required'],
            'message_parameter' => ['required'],
            'to_parameter' => ['required'],
            'sender_parameter' => ['required'],
            'base64_encoded_authentication' => ['required']
        ]);
        if ($request->authentication == "1") {

            $request->validate([
                'use_token' => ['required']
            ]);

            if ($request->use_token == "1") {
                $request->validate([
                    'token' => ['required'],
                    'use_token' => ['required'],
                ]);
            }else{
                $request->validate([
                    'auth_username' => ['required'],
                    'auth_password' => ['required']
                ]);
            }
        }
        try {

            $storeParam = $request->except('_token', 'parameter', 'value');

            if ($request->authentication == "1") {
                dd($request->use_token);

            }else{
                $storeParam['use_token'] = 0;
            }

            $parameters = $request->parameter;
            $ParameterValues = $request->value;

            $params = [];
            foreach ($parameters as $key => $value) {
                $params[$value] = $ParameterValues[$key];
            }
            $parameters = ["parameters" => json_encode($params)];
            $storeParam = array_replace($storeParam, $parameters);

            $storeParam['created_by'] = $request->user()->id;
            SmsApi::insert($storeParam);
            $this->resources['messages'] = array("type" => "success", "description" => "Sms api creation success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\SmsApi  $smsApi
     * @return \Illuminate\Http\Response
     */
    public function view(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'View SmsApi']);
        $this->resources['data'] = $this->getApiById($request);
        return view('admin.sms-api.sms_api_view')->with($this->resources);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\SmsApi  $smsApi
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Update SmsApi']);
        $this->resources['data'] = $this->getApiById($request);
        return view('admin.sms-api.sms_api_edit')->with($this->resources);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\SmsApi  $smsApi
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SmsApi $smsApi)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Update SmsApi']);

        $credentials = $request->validate([
            'id' => ['required'],
            'name' => ['required'],
            'url' => ['required', 'url'],
            'method' => ['required'],
            'authentication' => ['required'],
            'json_encoded' => ['required'],
            'api_type' => ['required'],
            'message_parameter' => ['required'],
            'to_parameter' => ['required'],
            'sender_parameter' => ['required'],
            'base64_encoded_authentication' => ['required']
        ]);
        if ($request->authentication == "1") {

            $request->validate([
                'use_token' => ['required']
            ]);

            if ($request->use_token == "1") {
                $request->validate([
                    'token' => ['required'],
                    'use_token' => ['required'],
                ]);
            }else{
                $request->validate([
                    'auth_username' => ['required'],
                    'auth_password' => ['required']
                ]);
            }
            
        }
        
        try {

            $storeParam = $request->except('_token', 'parameter', 'value');

            $parameters = $request->parameter;
            $ParameterValues = $request->value;

            $params = [];
            foreach ($parameters as $key => $value) {
                $params[$value] = $ParameterValues[$key];
            }

            $parameters = ["parameters" => json_encode($params)];
            $storeParam = array_replace($storeParam, $parameters);

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsApi::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->update($storeParam);
            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Sms api updated success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Sms api updated failed");
            }

            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\SmsApi  $smsApi
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request, SmsApi $smsApi)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Delete SmsApi']);
        $request->validate([
            'id' => ['required'],
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsApi::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if ($result) {
                $this->resources['messages'] = array("type" => "success", "description" => "Sms api deleted success");
            } else {
                $this->resources['messages'] = array("type" => "error", "description" => "Sms api deleted failed");
            }
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    public function getApi($request, $count)
    {
        try {

            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsApi::orderBy('id', 'desc');
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function getApiById($request)
    {
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = SmsApi::where('id', $request->id);
            if (!$superAdmin) {
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        } catch (\Exception $e) {
            throw $e;
        }
    }
}
