<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Contracts\Role;
use Spatie\Permission\Models\Role as ModelsRole;

class UserController extends Controller
{
    private $prefix = "admin/users/";
    private $resources = [];
    private $checkPermissionAndRolesController;
    public function __construct(CheckPermissionsAndRolesController $checkPermissionAndRolesController)
    {
        $this->resources['prefix'] = $this->prefix;
        $this->resources['roles'] = DB::table('roles')->select('id', 'name')->get();
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'View User']);
        $this->resources['users'] = $this->getUsers($request, 15);
        return view('admin.users.user_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = User::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
                if(!$superAdmin){
                    $order->where('created_by', $request->user()->id);
                }
                $order->orWhere('email', 'Like', '%' . $request->search . '%');
                $order->orWhere('teligram', 'Like', '%' . $request->search . '%');
               
                
            }
            $this->resources['users'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.users.user_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }
    public function addUserView(Request $request)
    {   
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Create User']);
        return view('admin.users.user_add')->with($this->resources);
    }
    public function editUserView(Request $request)
    {   
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Update User']);
        $this->resources['user'] = $this->getUsersById($request);
        return view('admin.users.user_edit')->with($this->resources);
    }

    public function view(Request $request)
    {   
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'View User']);
        $this->resources['user'] = $this->getUsersById($request);
        return view('admin.users.user_view')->with($this->resources);
    }

    public function store(Request $request)
    { 
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Create User']);
        $credentials = $request->validate([
            'name' => ['required'],
            'email' => ['required', 'email', 'unique:users'],
            'password' => ['required','required_with:password_confirmation','same:password_confirmation','min:6'],
            'password_confirmation' =>['required', 'min:6'],
            'teligram' =>['required'],
            'type' =>['required'],
            'status' =>['required'],
            'secret_code' =>['required']
        ]);
        try {
            
            $requestParam = $request->all();
            $password = ["password"=>Hash::make($requestParam['password'])];
            $requestParam = array_replace($requestParam, $password);
            $requestParam['created_by'] = Auth::user()->id;
            $user = User::create($requestParam);
            $role = ModelsRole::find($requestParam['type']);
            $user->syncRoles($role);
            Setting::create(["order_settings"=>["ip_auto_block"=>false, "link_auto_block"=>"false","redirect"=>false, "link_code_length"=>3], "user_id" => $user->id]);
            $this->resources['messages'] = array("type" => "success", "description"=>"User creation success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }
    public function update(Request $request)
    { 
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Update User']);
        $credentials = $request->validate([
            'name' => ['required'],
            'email' => ['required', 'email'], 
            'type' =>['required'],
            'id'=>['required'],
            'secret_code' =>['required']
        ]);
        if($request->password != ""){

            $credentials = $request->validate([
                'password' => ['required','required_with:password_confirmation','same:password_confirmation','min:6'],
                'password_confirmation' =>['required', 'min:6']
            ]);
        }
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $requestParam = $request->all();
            $query = User::where('id', $requestParam['id']);

            if($requestParam['password'] != ""){
                unset($requestParam['_token']);
                unset($requestParam['password_confirmation']);
                $requestParam['password'] = Hash::make($requestParam['password']);
                if(!$superAdmin){
                    $query->where('created_by', $request->user()->id);
                }
                $result = $query->update($requestParam);
            }else{
                if(!$superAdmin){
                    $query->where('created_by', $request->user()->id);
                }
                $result = $query->update([
                    'name'=>$requestParam['name'],
                    'email'=>$requestParam['email'],
                    'type'=>$requestParam['type'],
                    'status'=>$requestParam['status'],
                    'teligram'=>$requestParam['teligram'],
                    'secret_code'=>$requestParam['secret_code'],
                ]);
            }
            if($result){
                $user = User::find($requestParam['id']);
                $role = ModelsRole::find($requestParam['type']);
                $user->syncRoles($role);
                $this->resources['messages'] = array("type" => "success", "description"=>"User update success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"User update failed");
            }
            
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }

    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Delete User']);
        $request->validate([
            'id' => ['required'],   
        ]);
        try {
            $result = $this->deleteUserById($request);
            $this->resources['messages'] = array("type" => "success", "description"=>"User deleted success");
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
        
    }
    public function getAll()
    {
        try{
            $result = User::all();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function getUsers($request, $count)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = User::join('roles', 'users.type', 'roles.id')->select('users.*','roles.name as role_name')->orderBy('id','desc');
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function getUsersById($request)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = User::join('roles', 'users.type', 'roles.id')->select('users.*','roles.name as role_name');
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->where('users.id', $request->id)->firstOrFail();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
    public function deleteUserById($request)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = User::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }

    /**
     * @param array $data
     * @index email
     * @return boolean
     */
    public function checkUserActive(array $data)
    {
        try {
            $user = User::where('email', $data['email'])->first();
            if($user){
                if(isset($user->status) && $user->status == 1){
                    return true;
                }
            }
            return false;
        } catch (\Throwable $th) {
            return $th->getMessage();
        }
    }
}
