<?php

namespace App\Http\Controllers;

use App\Models\WhatsAppTemplate;
use Illuminate\Http\Request;

class WhatsAppTemplateController extends Controller
{
    private $prefix = "admin/whatsapp-template/";
    private $resources = [];
    private $checkPermissionAndRolesController;

    public function __construct(CheckPermissionsAndRolesController $checkPermissionAndRolesController)
    {
        $this->resources['prefix'] = $this->prefix;
        $this->checkPermissionAndRolesController = $checkPermissionAndRolesController;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Manage Sms Template']);
        $this->resources['data'] = $this->getTemplates($request, 15);
        return view('admin.whatsapp-template.ws_template_manage')->with($this->resources);
    }
    public function search(Request $request)
    {
        try {
            $request->validate([
                'search' => ['required'],
            ]);
            $order = WhatsAppTemplate::query();
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            if ($request->search) {
                $order->where('name', 'Like', '%' . $request->search . '%');
                if(!$superAdmin){
                    $order->where('created_by', $request->user()->id);
                }
            }
            $this->resources['data'] = $order->orderBy('id', 'DESC')->paginate(15);
            return view('admin.whatsapp-template.ws_template_manage')->with($this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create WhatsApp Template']);
        return view('admin.whatsapp-template.ws_template_add')->with($this->resources);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Create WhatsApp Template']);

        $credentials = $request->validate([
            'name' => ['required'],
            'template' => ['required'],

        ]);

        try {
            $storeParam = $request->except('_token');
            $storeParam['created_by'] = $request->user()->id;
            WhatsAppTemplate::create($storeParam);
            $this->resources['messages'] = array("type" => "success", "description" => "WhatsApp Template creation success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\WhatsAppTemplate  $wsTemplate
     * @return \Illuminate\Http\Response
     */
    public function view( Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'View WhatsApp Template']);
        $this->resources['data'] = $this->getTemplateById($request);
        return view('admin.whatsapp-template.ws_template_view')->with($this->resources);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\WhatsAppTemplate  $wsTemplate
     * @return \Illuminate\Http\Response
     */
    public function edit( Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Update WhatsApp Template']);
        $this->resources['data'] = $this->getTemplateById($request);
        return view('admin.whatsapp-template.ws_template_edit')->with($this->resources);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\WhatsAppTemplate  $wsTemplate
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission' => 'Update WhatsApp Template']);

        $credentials = $request->validate([
            'id'=>['required'],
            'name' => ['required'],
            'template' => ['required'],

        ]);

        try {
            $storeParam = $request->except('_token','id');
            $storeParam['created_by'] = $request->user()->id;
            WhatsAppTemplate::where('id', $request->id)->update($storeParam);
            $this->resources['messages'] = array("type" => "success", "description" => "WhatsApp Template updated success");
            return redirect()->back()->with('messages', $this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description" => $th->getMessage());
            return redirect()->back()->with('messages', $this->resources);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\WhatsAppTemplate  $wsTemplate
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $this->checkPermissionAndRolesController->permissionTo($request, ['permission'=>'Delete WhatsApp Template']);
        $request->validate([
            'id' => ['required'],   
        ]);
        try {
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppTemplate::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->delete();

            if($result){
                $this->resources['messages'] = array("type" => "success", "description"=>"WhatsApp Template deleted success");
            }else{
                $this->resources['messages'] = array("type" => "error", "description"=>"WhatsApp Template deleted failed");
            }
            return redirect()->back()->with('messages',$this->resources);
        } catch (\Throwable $th) {
            $this->resources['messages'] = array("type" => "error", "description"=>$th->getMessage());
            return redirect()->back()->with('messages',$this->resources);
        }
    }

    public function getTemplates($request, $count)
    {
        try{
           
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppTemplate::orderBy('id','desc');
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->paginate($count);
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }

    public function getTemplateById($request)
    {
        try{
            $superAdmin = $this->checkPermissionAndRolesController->superAdmin($request);
            $query = WhatsAppTemplate::where('id', $request->id);
            if(!$superAdmin){
                $query->where('created_by', $request->user()->id);
            }
            $result = $query->first();
            return $result;
        }catch(\Exception $e){
            throw $e;
        }
    }
}
